#region Copyright & License Information
/*
 * Copyright (c) The OpenRA Developers and Contributors
 * This file is part of OpenRA, which is free software. It is made
 * available to you under the terms of the GNU General Public License
 * as published by the Free Software Foundation, either version 3 of
 * the License, or (at your option) any later version. For more
 * information, see COPYING.
 */
#endregion

using System.Collections.Generic;

namespace OpenRA
{
	// List of keycodes. Duplicated from SDL 2.0.1, with the addition
	// of MOUSE4 and MOUSE5.
	public enum Keycode
	{
		UNKNOWN = 0,
		RETURN = '\r',
		ESCAPE = 27,
		BACKSPACE = '\b',
		TAB = '\t',
		SPACE = ' ',
		EXCLAIM = '!',
		QUOTEDBL = '"',
		HASH = '#',
		PERCENT = '%',
		DOLLAR = '$',
		AMPERSAND = '&',
		QUOTE = '\'',
		LEFTPAREN = '(',
		RIGHTPAREN = ')',
		ASTERISK = '*',
		PLUS = '+',
		COMMA = ',',
		MINUS = '-',
		PERIOD = '.',
		SLASH = '/',
		NUMBER_0 = '0',
		NUMBER_1 = '1',
		NUMBER_2 = '2',
		NUMBER_3 = '3',
		NUMBER_4 = '4',
		NUMBER_5 = '5',
		NUMBER_6 = '6',
		NUMBER_7 = '7',
		NUMBER_8 = '8',
		NUMBER_9 = '9',
		COLON = ':',
		SEMICOLON = ';',
		LESS = '<',
		EQUALS = '=',
		GREATER = '>',
		QUESTION = '?',
		AT = '@',
		LEFTBRACKET = '[',
		BACKSLASH = '\\',
		RIGHTBRACKET = ']',
		CARET = '^',
		UNDERSCORE = '_',
		BACKQUOTE = '`',
		A = 'a',
		B = 'b',
		C = 'c',
		D = 'd',
		E = 'e',
		F = 'f',
		G = 'g',
		H = 'h',
		I = 'i',
		J = 'j',
		K = 'k',
		L = 'l',
		M = 'm',
		N = 'n',
		O = 'o',
		P = 'p',
		Q = 'q',
		R = 'r',
		S = 's',
		T = 't',
		U = 'u',
		V = 'v',
		W = 'w',
		X = 'x',
		Y = 'y',
		Z = 'z',
		CAPSLOCK = 57 | (1 << 30),
		F1 = 58 | (1 << 30),
		F2 = 59 | (1 << 30),
		F3 = 60 | (1 << 30),
		F4 = 61 | (1 << 30),
		F5 = 62 | (1 << 30),
		F6 = 63 | (1 << 30),
		F7 = 64 | (1 << 30),
		F8 = 65 | (1 << 30),
		F9 = 66 | (1 << 30),
		F10 = 67 | (1 << 30),
		F11 = 68 | (1 << 30),
		F12 = 69 | (1 << 30),
		PRINTSCREEN = 70 | (1 << 30),
		SCROLLLOCK = 71 | (1 << 30),
		PAUSE = 72 | (1 << 30),
		INSERT = 73 | (1 << 30),
		HOME = 74 | (1 << 30),
		PAGEUP = 75 | (1 << 30),
		DELETE = 127,
		END = 77 | (1 << 30),
		PAGEDOWN = 78 | (1 << 30),
		RIGHT = 79 | (1 << 30),
		LEFT = 80 | (1 << 30),
		DOWN = 81 | (1 << 30),
		UP = 82 | (1 << 30),
		NUMLOCKCLEAR = 83 | (1 << 30),
		KP_DIVIDE = 84 | (1 << 30),
		KP_MULTIPLY = 85 | (1 << 30),
		KP_MINUS = 86 | (1 << 30),
		KP_PLUS = 87 | (1 << 30),
		KP_ENTER = 88 | (1 << 30),
		KP_1 = 89 | (1 << 30),
		KP_2 = 90 | (1 << 30),
		KP_3 = 91 | (1 << 30),
		KP_4 = 92 | (1 << 30),
		KP_5 = 93 | (1 << 30),
		KP_6 = 94 | (1 << 30),
		KP_7 = 95 | (1 << 30),
		KP_8 = 96 | (1 << 30),
		KP_9 = 97 | (1 << 30),
		KP_0 = 98 | (1 << 30),
		KP_PERIOD = 99 | (1 << 30),
		APPLICATION = 101 | (1 << 30),
		POWER = 102 | (1 << 30),
		KP_EQUALS = 103 | (1 << 30),
		F13 = 104 | (1 << 30),
		F14 = 105 | (1 << 30),
		F15 = 106 | (1 << 30),
		F16 = 107 | (1 << 30),
		F17 = 108 | (1 << 30),
		F18 = 109 | (1 << 30),
		F19 = 110 | (1 << 30),
		F20 = 111 | (1 << 30),
		F21 = 112 | (1 << 30),
		F22 = 113 | (1 << 30),
		F23 = 114 | (1 << 30),
		F24 = 115 | (1 << 30),
		EXECUTE = 116 | (1 << 30),
		HELP = 117 | (1 << 30),
		MENU = 118 | (1 << 30),
		SELECT = 119 | (1 << 30),
		STOP = 120 | (1 << 30),
		AGAIN = 121 | (1 << 30),
		UNDO = 122 | (1 << 30),
		CUT = 123 | (1 << 30),
		COPY = 124 | (1 << 30),
		PASTE = 125 | (1 << 30),
		FIND = 126 | (1 << 30),
		MUTE = 127 | (1 << 30),
		VOLUMEUP = 128 | (1 << 30),
		VOLUMEDOWN = 129 | (1 << 30),
		KP_COMMA = 133 | (1 << 30),
		KP_EQUALSAS400 = 134 | (1 << 30),
		ALTERASE = 153 | (1 << 30),
		SYSREQ = 154 | (1 << 30),
		CANCEL = 155 | (1 << 30),
		CLEAR = 156 | (1 << 30),
		PRIOR = 157 | (1 << 30),
		RETURN2 = 158 | (1 << 30),
		SEPARATOR = 159 | (1 << 30),
		OUT = 160 | (1 << 30),
		OPER = 161 | (1 << 30),
		CLEARAGAIN = 162 | (1 << 30),
		CRSEL = 163 | (1 << 30),
		EXSEL = 164 | (1 << 30),
		KP_00 = 176 | (1 << 30),
		KP_000 = 177 | (1 << 30),
		THOUSANDSSEPARATOR = 178 | (1 << 30),
		DECIMALSEPARATOR = 179 | (1 << 30),
		CURRENCYUNIT = 180 | (1 << 30),
		CURRENCYSUBUNIT = 181 | (1 << 30),
		KP_LEFTPAREN = 182 | (1 << 30),
		KP_RIGHTPAREN = 183 | (1 << 30),
		KP_LEFTBRACE = 184 | (1 << 30),
		KP_RIGHTBRACE = 185 | (1 << 30),
		KP_TAB = 186 | (1 << 30),
		KP_BACKSPACE = 187 | (1 << 30),
		KP_A = 188 | (1 << 30),
		KP_B = 189 | (1 << 30),
		KP_C = 190 | (1 << 30),
		KP_D = 191 | (1 << 30),
		KP_E = 192 | (1 << 30),
		KP_F = 193 | (1 << 30),
		KP_XOR = 194 | (1 << 30),
		KP_POWER = 195 | (1 << 30),
		KP_PERCENT = 196 | (1 << 30),
		KP_LESS = 197 | (1 << 30),
		KP_GREATER = 198 | (1 << 30),
		KP_AMPERSAND = 199 | (1 << 30),
		KP_DBLAMPERSAND = 200 | (1 << 30),
		KP_VERTICALBAR = 201 | (1 << 30),
		KP_DBLVERTICALBAR = 202 | (1 << 30),
		KP_COLON = 203 | (1 << 30),
		KP_HASH = 204 | (1 << 30),
		KP_SPACE = 205 | (1 << 30),
		KP_AT = 206 | (1 << 30),
		KP_EXCLAM = 207 | (1 << 30),
		KP_MEMSTORE = 208 | (1 << 30),
		KP_MEMRECALL = 209 | (1 << 30),
		KP_MEMCLEAR = 210 | (1 << 30),
		KP_MEMADD = 211 | (1 << 30),
		KP_MEMSUBTRACT = 212 | (1 << 30),
		KP_MEMMULTIPLY = 213 | (1 << 30),
		KP_MEMDIVIDE = 214 | (1 << 30),
		KP_PLUSMINUS = 215 | (1 << 30),
		KP_CLEAR = 216 | (1 << 30),
		KP_CLEARENTRY = 217 | (1 << 30),
		KP_BINARY = 218 | (1 << 30),
		KP_OCTAL = 219 | (1 << 30),
		KP_DECIMAL = 220 | (1 << 30),
		KP_HEXADECIMAL = 221 | (1 << 30),
		LCTRL = 224 | (1 << 30),
		LSHIFT = 225 | (1 << 30),
		LALT = 226 | (1 << 30),
		LGUI = 227 | (1 << 30),
		RCTRL = 228 | (1 << 30),
		RSHIFT = 229 | (1 << 30),
		RALT = 230 | (1 << 30),
		RGUI = 231 | (1 << 30),
		MODE = 257 | (1 << 30),
		AUDIONEXT = 258 | (1 << 30),
		AUDIOPREV = 259 | (1 << 30),
		AUDIOSTOP = 260 | (1 << 30),
		AUDIOPLAY = 261 | (1 << 30),
		AUDIOMUTE = 262 | (1 << 30),
		MEDIASELECT = 263 | (1 << 30),
		WWW = 264 | (1 << 30),
		MAIL = 265 | (1 << 30),
		CALCULATOR = 266 | (1 << 30),
		COMPUTER = 267 | (1 << 30),
		AC_SEARCH = 268 | (1 << 30),
		AC_HOME = 269 | (1 << 30),
		AC_BACK = 270 | (1 << 30),
		AC_FORWARD = 271 | (1 << 30),
		AC_STOP = 272 | (1 << 30),
		AC_REFRESH = 273 | (1 << 30),
		AC_BOOKMARKS = 274 | (1 << 30),
		BRIGHTNESSDOWN = 275 | (1 << 30),
		BRIGHTNESSUP = 276 | (1 << 30),
		DISPLAYSWITCH = 277 | (1 << 30),
		KBDILLUMTOGGLE = 278 | (1 << 30),
		KBDILLUMDOWN = 279 | (1 << 30),
		KBDILLUMUP = 280 | (1 << 30),
		EJECT = 281 | (1 << 30),
		SLEEP = 282 | (1 << 30),
		MOUSE4 = 283 | (1 << 30),
		MOUSE5 = 284 | (1 << 30)
	}

	public static class KeycodeExts
	{
		[FluentReference(Traits.LintDictionaryReference.Values)]
		public static readonly IReadOnlyDictionary<Keycode, string> KeycodeFluentKeys = new Dictionary<Keycode, string>()
		{
			{ Keycode.UNKNOWN, "keycode.unknown" },
			{ Keycode.RETURN, "keycode.return" },
			{ Keycode.ESCAPE, "keycode.escape" },
			{ Keycode.BACKSPACE, "keycode.backspace" },
			{ Keycode.TAB, "keycode.tab" },
			{ Keycode.SPACE, "keycode.space" },
			{ Keycode.EXCLAIM, "keycode.exclaim" },
			{ Keycode.QUOTEDBL, "keycode.quotedbl" },
			{ Keycode.HASH, "keycode.hash" },
			{ Keycode.PERCENT, "keycode.percent" },
			{ Keycode.DOLLAR, "keycode.dollar" },
			{ Keycode.AMPERSAND, "keycode.ampersand" },
			{ Keycode.QUOTE, "keycode.quote" },
			{ Keycode.LEFTPAREN, "keycode.leftparen" },
			{ Keycode.RIGHTPAREN, "keycode.rightparen" },
			{ Keycode.ASTERISK, "keycode.asterisk" },
			{ Keycode.PLUS, "keycode.plus" },
			{ Keycode.COMMA, "keycode.comma" },
			{ Keycode.MINUS, "keycode.minus" },
			{ Keycode.PERIOD, "keycode.period" },
			{ Keycode.SLASH, "keycode.slash" },
			{ Keycode.NUMBER_0, "keycode.number_0" },
			{ Keycode.NUMBER_1, "keycode.number_1" },
			{ Keycode.NUMBER_2, "keycode.number_2" },
			{ Keycode.NUMBER_3, "keycode.number_3" },
			{ Keycode.NUMBER_4, "keycode.number_4" },
			{ Keycode.NUMBER_5, "keycode.number_5" },
			{ Keycode.NUMBER_6, "keycode.number_6" },
			{ Keycode.NUMBER_7, "keycode.number_7" },
			{ Keycode.NUMBER_8, "keycode.number_8" },
			{ Keycode.NUMBER_9, "keycode.number_9" },
			{ Keycode.COLON, "keycode.colon" },
			{ Keycode.SEMICOLON, "keycode.semicolon" },
			{ Keycode.LESS, "keycode.less" },
			{ Keycode.EQUALS, "keycode.equals" },
			{ Keycode.GREATER, "keycode.greater" },
			{ Keycode.QUESTION, "keycode.question" },
			{ Keycode.AT, "keycode.at" },
			{ Keycode.LEFTBRACKET, "keycode.leftbracket" },
			{ Keycode.BACKSLASH, "keycode.backslash" },
			{ Keycode.RIGHTBRACKET, "keycode.rightbracket" },
			{ Keycode.CARET, "keycode.caret" },
			{ Keycode.UNDERSCORE, "keycode.underscore" },
			{ Keycode.BACKQUOTE, "keycode.backquote" },
			{ Keycode.A, "keycode.a" },
			{ Keycode.B, "keycode.b" },
			{ Keycode.C, "keycode.c" },
			{ Keycode.D, "keycode.d" },
			{ Keycode.E, "keycode.e" },
			{ Keycode.F, "keycode.f" },
			{ Keycode.G, "keycode.g" },
			{ Keycode.H, "keycode.h" },
			{ Keycode.I, "keycode.i" },
			{ Keycode.J, "keycode.j" },
			{ Keycode.K, "keycode.k" },
			{ Keycode.L, "keycode.l" },
			{ Keycode.M, "keycode.m" },
			{ Keycode.N, "keycode.n" },
			{ Keycode.O, "keycode.o" },
			{ Keycode.P, "keycode.p" },
			{ Keycode.Q, "keycode.q" },
			{ Keycode.R, "keycode.r" },
			{ Keycode.S, "keycode.s" },
			{ Keycode.T, "keycode.t" },
			{ Keycode.U, "keycode.u" },
			{ Keycode.V, "keycode.v" },
			{ Keycode.W, "keycode.w" },
			{ Keycode.X, "keycode.x" },
			{ Keycode.Y, "keycode.y" },
			{ Keycode.Z, "keycode.z" },
			{ Keycode.CAPSLOCK, "keycode.capslock" },
			{ Keycode.F1, "keycode.f1" },
			{ Keycode.F2, "keycode.f2" },
			{ Keycode.F3, "keycode.f3" },
			{ Keycode.F4, "keycode.f4" },
			{ Keycode.F5, "keycode.f5" },
			{ Keycode.F6, "keycode.f6" },
			{ Keycode.F7, "keycode.f7" },
			{ Keycode.F8, "keycode.f8" },
			{ Keycode.F9, "keycode.f9" },
			{ Keycode.F10, "keycode.f10" },
			{ Keycode.F11, "keycode.f11" },
			{ Keycode.F12, "keycode.f12" },
			{ Keycode.PRINTSCREEN, "keycode.printscreen" },
			{ Keycode.SCROLLLOCK, "keycode.scrolllock" },
			{ Keycode.PAUSE, "keycode.pause" },
			{ Keycode.INSERT, "keycode.insert" },
			{ Keycode.HOME, "keycode.home" },
			{ Keycode.PAGEUP, "keycode.pageup" },
			{ Keycode.DELETE, "keycode.delete" },
			{ Keycode.END, "keycode.end" },
			{ Keycode.PAGEDOWN, "keycode.pagedown" },
			{ Keycode.RIGHT, "keycode.right" },
			{ Keycode.LEFT, "keycode.left" },
			{ Keycode.DOWN, "keycode.down" },
			{ Keycode.UP, "keycode.up" },
			{ Keycode.NUMLOCKCLEAR, "keycode.numlockclear" },
			{ Keycode.KP_DIVIDE, "keycode.kp_divide" },
			{ Keycode.KP_MULTIPLY, "keycode.kp_multiply" },
			{ Keycode.KP_MINUS, "keycode.kp_minus" },
			{ Keycode.KP_PLUS, "keycode.kp_plus" },
			{ Keycode.KP_ENTER, "keycode.kp_enter" },
			{ Keycode.KP_1, "keycode.kp_1" },
			{ Keycode.KP_2, "keycode.kp_2" },
			{ Keycode.KP_3, "keycode.kp_3" },
			{ Keycode.KP_4, "keycode.kp_4" },
			{ Keycode.KP_5, "keycode.kp_5" },
			{ Keycode.KP_6, "keycode.kp_6" },
			{ Keycode.KP_7, "keycode.kp_7" },
			{ Keycode.KP_8, "keycode.kp_8" },
			{ Keycode.KP_9, "keycode.kp_9" },
			{ Keycode.KP_0, "keycode.kp_0" },
			{ Keycode.KP_PERIOD, "keycode.kp_period" },
			{ Keycode.APPLICATION, "keycode.application" },
			{ Keycode.POWER, "keycode.power" },
			{ Keycode.KP_EQUALS, "keycode.kp_equals" },
			{ Keycode.F13, "keycode.f13" },
			{ Keycode.F14, "keycode.f14" },
			{ Keycode.F15, "keycode.f15" },
			{ Keycode.F16, "keycode.f16" },
			{ Keycode.F17, "keycode.f17" },
			{ Keycode.F18, "keycode.f18" },
			{ Keycode.F19, "keycode.f19" },
			{ Keycode.F20, "keycode.f20" },
			{ Keycode.F21, "keycode.f21" },
			{ Keycode.F22, "keycode.f22" },
			{ Keycode.F23, "keycode.f23" },
			{ Keycode.F24, "keycode.f24" },
			{ Keycode.EXECUTE, "keycode.execute" },
			{ Keycode.HELP, "keycode.help" },
			{ Keycode.MENU, "keycode.menu" },
			{ Keycode.SELECT, "keycode.select" },
			{ Keycode.STOP, "keycode.stop" },
			{ Keycode.AGAIN, "keycode.again" },
			{ Keycode.UNDO, "keycode.undo" },
			{ Keycode.CUT, "keycode.cut" },
			{ Keycode.COPY, "keycode.copy" },
			{ Keycode.PASTE, "keycode.paste" },
			{ Keycode.FIND, "keycode.find" },
			{ Keycode.MUTE, "keycode.mute" },
			{ Keycode.VOLUMEUP, "keycode.volumeup" },
			{ Keycode.VOLUMEDOWN, "keycode.volumedown" },
			{ Keycode.KP_COMMA, "keycode.kp_comma" },
			{ Keycode.KP_EQUALSAS400, "keycode.kp_equalsas400" },
			{ Keycode.ALTERASE, "keycode.alterase" },
			{ Keycode.SYSREQ, "keycode.sysreq" },
			{ Keycode.CANCEL, "keycode.cancel" },
			{ Keycode.CLEAR, "keycode.clear" },
			{ Keycode.PRIOR, "keycode.prior" },
			{ Keycode.RETURN2, "keycode.return2" },
			{ Keycode.SEPARATOR, "keycode.separator" },
			{ Keycode.OUT, "keycode.out" },
			{ Keycode.OPER, "keycode.oper" },
			{ Keycode.CLEARAGAIN, "keycode.clearagain" },
			{ Keycode.CRSEL, "keycode.crsel" },
			{ Keycode.EXSEL, "keycode.exsel" },
			{ Keycode.KP_00, "keycode.kp_00" },
			{ Keycode.KP_000, "keycode.kp_000" },
			{ Keycode.THOUSANDSSEPARATOR, "keycode.thousandsseparator" },
			{ Keycode.DECIMALSEPARATOR, "keycode.decimalseparator" },
			{ Keycode.CURRENCYUNIT, "keycode.currencyunit" },
			{ Keycode.CURRENCYSUBUNIT, "keycode.currencysubunit" },
			{ Keycode.KP_LEFTPAREN, "keycode.kp_leftparen" },
			{ Keycode.KP_RIGHTPAREN, "keycode.kp_rightparen" },
			{ Keycode.KP_LEFTBRACE, "keycode.kp_leftbrace" },
			{ Keycode.KP_RIGHTBRACE, "keycode.kp_rightbrace" },
			{ Keycode.KP_TAB, "keycode.kp_tab" },
			{ Keycode.KP_BACKSPACE, "keycode.kp_backspace" },
			{ Keycode.KP_A, "keycode.kp_a" },
			{ Keycode.KP_B, "keycode.kp_b" },
			{ Keycode.KP_C, "keycode.kp_c" },
			{ Keycode.KP_D, "keycode.kp_d" },
			{ Keycode.KP_E, "keycode.kp_e" },
			{ Keycode.KP_F, "keycode.kp_f" },
			{ Keycode.KP_XOR, "keycode.kp_xor" },
			{ Keycode.KP_POWER, "keycode.kp_power" },
			{ Keycode.KP_PERCENT, "keycode.kp_percent" },
			{ Keycode.KP_LESS, "keycode.kp_less" },
			{ Keycode.KP_GREATER, "keycode.kp_greater" },
			{ Keycode.KP_AMPERSAND, "keycode.kp_ampersand" },
			{ Keycode.KP_DBLAMPERSAND, "keycode.kp_dblampersand" },
			{ Keycode.KP_VERTICALBAR, "keycode.kp_verticalbar" },
			{ Keycode.KP_DBLVERTICALBAR, "keycode.kp_dblverticalbar" },
			{ Keycode.KP_COLON, "keycode.kp_colon" },
			{ Keycode.KP_HASH, "keycode.kp_hash" },
			{ Keycode.KP_SPACE, "keycode.kp_space" },
			{ Keycode.KP_AT, "keycode.kp_at" },
			{ Keycode.KP_EXCLAM, "keycode.kp_exclam" },
			{ Keycode.KP_MEMSTORE, "keycode.kp_memstore" },
			{ Keycode.KP_MEMRECALL, "keycode.kp_memrecall" },
			{ Keycode.KP_MEMCLEAR, "keycode.kp_memclear" },
			{ Keycode.KP_MEMADD, "keycode.kp_memadd" },
			{ Keycode.KP_MEMSUBTRACT, "keycode.kp_memsubtract" },
			{ Keycode.KP_MEMMULTIPLY, "keycode.kp_memmultiply" },
			{ Keycode.KP_MEMDIVIDE, "keycode.kp_memdivide" },
			{ Keycode.KP_PLUSMINUS, "keycode.kp_plusminus" },
			{ Keycode.KP_CLEAR, "keycode.kp_clear" },
			{ Keycode.KP_CLEARENTRY, "keycode.kp_clearentry" },
			{ Keycode.KP_BINARY, "keycode.kp_binary" },
			{ Keycode.KP_OCTAL, "keycode.kp_octal" },
			{ Keycode.KP_DECIMAL, "keycode.kp_decimal" },
			{ Keycode.KP_HEXADECIMAL, "keycode.kp_hexadecimal" },
			{ Keycode.LCTRL, "keycode.lctrl" },
			{ Keycode.LSHIFT, "keycode.lshift" },
			{ Keycode.LALT, "keycode.lalt" },
			{ Keycode.LGUI, "keycode.lgui" },
			{ Keycode.RCTRL, "keycode.rctrl" },
			{ Keycode.RSHIFT, "keycode.rshift" },
			{ Keycode.RALT, "keycode.ralt" },
			{ Keycode.RGUI, "keycode.rgui" },
			{ Keycode.MODE, "keycode.mode" },
			{ Keycode.AUDIONEXT, "keycode.audionext" },
			{ Keycode.AUDIOPREV, "keycode.audioprev" },
			{ Keycode.AUDIOSTOP, "keycode.audiostop" },
			{ Keycode.AUDIOPLAY, "keycode.audioplay" },
			{ Keycode.AUDIOMUTE, "keycode.audiomute" },
			{ Keycode.MEDIASELECT, "keycode.mediaselect" },
			{ Keycode.WWW, "keycode.www" },
			{ Keycode.MAIL, "keycode.mail" },
			{ Keycode.CALCULATOR, "keycode.calculator" },
			{ Keycode.COMPUTER, "keycode.computer" },
			{ Keycode.AC_SEARCH, "keycode.ac_search" },
			{ Keycode.AC_HOME, "keycode.ac_home" },
			{ Keycode.AC_BACK, "keycode.ac_back" },
			{ Keycode.AC_FORWARD, "keycode.ac_forward" },
			{ Keycode.AC_STOP, "keycode.ac_stop" },
			{ Keycode.AC_REFRESH, "keycode.ac_refresh" },
			{ Keycode.AC_BOOKMARKS, "keycode.ac_bookmarks" },
			{ Keycode.BRIGHTNESSDOWN, "keycode.brightnessdown" },
			{ Keycode.BRIGHTNESSUP, "keycode.brightnessup" },
			{ Keycode.DISPLAYSWITCH, "keycode.displayswitch" },
			{ Keycode.KBDILLUMTOGGLE, "keycode.kbdillumtoggle" },
			{ Keycode.KBDILLUMDOWN, "keycode.kbdillumdown" },
			{ Keycode.KBDILLUMUP, "keycode.kbdillumup" },
			{ Keycode.EJECT, "keycode.eject" },
			{ Keycode.SLEEP, "keycode.sleep" },
			{ Keycode.MOUSE4, "keycode.mouse4" },
			{ Keycode.MOUSE5, "keycode.mouse5" },
		};

		public static string DisplayString(Keycode k)
		{
			if (!KeycodeFluentKeys.TryGetValue(k, out var fluentKey))
				return k.ToString();

			return FluentProvider.GetMessage(fluentKey);
		}
	}
}
